<?php
// send_sms_api.php
// Full working SMS send endpoint with per-user and default overrides.
// Place this file where your API requests will hit. It expects db.php to provide $conn (mysqli).

error_reporting(E_ALL);
ini_set('display_errors', 1);
header('Content-Type: application/json; charset=utf-8');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

// OPTIONS preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

date_default_timezone_set('Asia/Dhaka');

// include DB connection (must set $conn = new mysqli(...))
require_once __DIR__ . '/db.php';
if (!isset($conn) || !($conn instanceof mysqli)) {
    echo json_encode(['success'=>false, 'message'=>'Database connection missing.']);
    exit;
}
$conn->query("SET time_zone = '+06:00'");

// Read JSON body
$raw = file_get_contents("php://input");
$data = json_decode($raw, true);
if (!is_array($data)) {
    echo json_encode(['success'=>false, 'message'=>'Invalid JSON body.']);
    exit;
}

// Input fields
$api_key     = trim($data['api'] ?? '');
$numbers_raw = trim($data['number'] ?? '');
$message_raw = trim($data['message'] ?? '');

// Basic validation
$errors = [];
if ($api_key === '') $errors[] = 'api field is required.';
if ($numbers_raw === '') $errors[] = 'number field is required.';
if ($message_raw === '') $errors[] = 'message field is required.';
if (!empty($errors)) {
    echo json_encode(['success'=>false, 'message'=>implode(' ', $errors)]);
    exit;
}

// Normalize numbers: split by comma, newline, semicolon
$numbers = preg_split('/[,\n;\r]+/', $numbers_raw);
$numbers = array_map('trim', $numbers);
$numbers = array_filter($numbers, function($n){ return $n !== ''; });

// remove non digit characters and filter length
$clean_numbers = [];
foreach ($numbers as $n) {
    $n2 = preg_replace('/\D+/', '', $n);
    // simple local number length check (>=10) - adapt if you need different validation
    if (strlen($n2) >= 10) $clean_numbers[] = $n2;
}
if (count($clean_numbers) === 0) {
    echo json_encode(['success'=>false, 'message'=>'No valid numbers found.']);
    exit;
}

// Find user by API key
$stmt = $conn->prepare("SELECT * FROM users WHERE api_key = ? LIMIT 1");
$stmt->bind_param("s", $api_key);
$stmt->execute();
$user_res = $stmt->get_result();
$stmt->close();

if (!$user_res || $user_res->num_rows === 0) {
    echo json_encode(['success'=>false, 'message'=>'Invalid API key.']);
    exit;
}
$user = $user_res->fetch_assoc();
$email = $user['email'] ?? null;

// Check account status/expiry if you store them
if (isset($user['status']) && $user['status'] !== 'active') {
    echo json_encode(['success'=>false, 'message'=>'Account blocked.']);
    exit;
}
if (isset($user['expiration_time']) && intval($user['expiration_time']) > 0 && time() > intval($user['expiration_time'])) {
    echo json_encode(['success'=>false, 'message'=>'API access expired.']);
    exit;
}

// Load defaults from settings table (id=1)
$settings = $conn->query("SELECT * FROM settings WHERE id = 1")->fetch_assoc() ?? [];
$default_rate = isset($settings['default_sms_rate']) ? floatval($settings['default_sms_rate']) : 0.40; // fallback 0.40
$default_api  = isset($settings['default_sms_api_url']) ? $settings['default_sms_api_url'] : "http://bulksmsbd.net/api/smsapi?api_key=PGH6awSDlG2SvKfgcU9A&type=text&number=[Number]&senderid=&message=[Message]";

// Per-user overrides (if set and valid)
$user_rate = (isset($user['sms_rate']) && floatval($user['sms_rate']) > 0) ? floatval($user['sms_rate']) : null;
$user_api  = (isset($user['sms_api_url']) && strlen(trim($user['sms_api_url'])) > 5) ? trim($user['sms_api_url']) : null;

$final_rate = ($user_rate !== null) ? $user_rate : $default_rate;
$api_template = ($user_api !== null) ? $user_api : $default_api;

// Helper: detect Unicode (non-ascii) characters => use 70 char per segment
function requires_unicode($text) {
    // return true if any non-ASCII char present
    return (bool) preg_match('/[^\x00-\x7F]/u', $text);
}

// Count characters correctly with mb_strlen
$encoding_is_unicode = requires_unicode($message_raw);
$char_limit = $encoding_is_unicode ? 70 : 160;
$char_count = mb_strlen($message_raw, 'UTF-8');
$segments = max(1, (int) ceil($char_count / $char_limit));

$total_numbers = count($clean_numbers);
$total_cost = $segments * $total_numbers * $final_rate;

// Check balance
$current_balance = isset($user['request_limit']) ? floatval($user['request_limit']) : 0.0;
if ($current_balance < $total_cost) {
    echo json_encode([
        'success'=>false,
        'message'=>'Insufficient balance.',
        'required'=>$total_cost,
        'available'=>$current_balance
    ]);
    exit;
}

// Prepare transaction: log each SMS and deduct balance atomically
$sent_count = 0;
$failed = [];
$logs = [];

$conn->begin_transaction(MYSQLI_TRANS_START_READ_WRITE);

try {
    // Prepared insert for logs
    $insert_stmt = $conn->prepare("INSERT INTO sms_logs (email, number, message, segments, cost, api_key, api_used, encoding_type, sent_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
    if (!$insert_stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }

    // Iterate and send
    foreach ($clean_numbers as $num) {
        // Build final url by replacing placeholders
        $final_url = str_replace(
            ['[Number]','[Message]'],
            [urlencode($num), urlencode($message_raw)],
            $api_template
        );

        // If API template doesn't contain [Number] and [Message], we still call final_url as-is.
        // Use curl to call API (GET). If your API requires POST, change here accordingly.
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $final_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        // Optionally: set User-Agent or headers if provider needs them
        $response = curl_exec($ch);
        $curl_errno = curl_errno($ch);
        $curl_err = curl_error($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        // Decide success by provider response. This is provider-specific.
        // We'll treat non-empty response and HTTP 200 as success (adjust per provider).
        $is_success = ($curl_errno === 0 && $http_code >= 200 && $http_code < 300 && $response !== false && strlen(trim((string)$response)) > 0);

        // Cost for this number
        $cost_for_number = $segments * $final_rate;

        if ($is_success) {
            // insert log row
            $encoding_label = $encoding_is_unicode ? 'UTF-8' : 'GSM_7BIT';
            $insert_stmt->bind_param("sssidsss", $email, $num, $message_raw, $segments, $cost_for_number, $api_key, $api_template, $encoding_label);
            // NOTE: bind_param format: "sssidsss" but we have 8 columns -> ensure types match: s string, i int, d double
            // We'll use explicit types with correct order: email(s), number(s), message(s), segments(i), cost(d), api_key(s), api_used(s), encoding_type(s)
            // Because mysqli bind_param needs matching types, re-prepare with correct types below if necessary.
            // To avoid mismatch, re-prepare with correct format string:
        } else {
            $failed[] = [
                'number' => $num,
                'http_code' => $http_code,
                'curl_errno' => $curl_errno,
                'curl_err' => $curl_err,
                'response' => $response
            ];
        }

        // Because of the binding type detail above, do actual insert using a safe approach:
        if ($is_success) {
            $ins = $conn->prepare("INSERT INTO sms_logs (email, number, message, segments, cost, api_key, api_used, encoding_type, sent_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
            if (!$ins) throw new Exception("Prepare insert failed: " . $conn->error);
            $encoding_label = $encoding_is_unicode ? 'UTF-8' : 'GSM_7BIT';
            $ins->bind_param("sssiddss", $email, $num, $message_raw, $segments, $cost_for_number, $api_key, $api_template, $encoding_label);
            if (!$ins->execute()) {
                // treat execution failure as failure
                $failed[] = ['number'=>$num, 'error'=>$ins->error];
                $ins->close();
            } else {
                $sent_count++;
                $logs[] = ['number'=>$num, 'cost'=>$cost_for_number, 'id'=>$ins->insert_id];
                $ins->close();
            }
        }
    } // end foreach numbers

    // Deduct balance once
    $new_balance = $current_balance - $total_cost;
    $upd = $conn->prepare("UPDATE users SET request_limit = ? WHERE email = ?");
    if (!$upd) throw new Exception("Prepare update failed: " . $conn->error);
    $upd->bind_param("ds", $new_balance, $email);
    if (!$upd->execute()) throw new Exception("Balance update failed: " . $upd->error);
    $upd->close();

    $conn->commit();

    echo json_encode([
        'success'=>true,
        'message'=>'SMS processing finished.',
        'sent'=>$sent_count,
        'failed_count'=>count($failed),
        'failed'=>$failed,
        'logs'=>$logs,
        'total_numbers'=>$total_numbers,
        'segments'=>$segments,
        'rate'=>$final_rate,
        'total_cost'=>$total_cost,
        'new_balance'=>$new_balance,
        'encoding'=>$encoding_is_unicode ? 'UTF-8' : 'GSM_7BIT'
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode([
        'success'=>false,
        'message'=>'Exception: ' . $e->getMessage(),
        'trace'=> $e->getTraceAsString()
    ]);
    exit;
}
